﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_VALowVisionSurvey : BaseTBIInstrumentPage {
    #region Variable Declarations

    private const string FailedFormValuesName = "LowVisionFailedFormValues";
    private const string DeferredMessageKeyName = "LowVisionDeferredMessageKey";
    private const string BaselineText = "baseline";
    protected int _surveyTypeId = 17; // Database Id for VA Low Vision Visual Functioning (VA LV VFQ 20) Survey

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    SURVEY _survey;   // the root object in the Survey set

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();
    #endregion Variable Declarations

    #region Public Methods

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_VALowVisionSurvey() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }
    #endregion Public Methods

    #region Protected Methods

    protected void Page_Init(object sender, EventArgs e)
    {
        VeteransAffairs.Registries.QuestionChoiceRetriever.Reset(_tbiInstrumentManager, _surveyTypeId);
    }

    protected void Page_Unload(object sender, EventArgs e)
    {
        VeteransAffairs.Registries.QuestionChoiceRetriever.Reset(null, -1);
    }

    bool preventNotification = false;

    protected void Page_PreRenderComplete(object sender, EventArgs e)
    {
        Dictionary<string, string> cache;
        if (!Page.IsPostBack && Helpers.SetupPageFromCache(this, FailedFormValuesName, out cache))
        {
            RestoreDateFromCache("D", cache);
            SuppressSaveMessage = false;
            SaveFailMessage = "Unable to save the survey.\r\n\r\nThere may be network issues preventing you from saving.\r\n\r\nPlease try to submit your changes again.";
            RenderStartUpMessage(false);
        }
        else if (Page.IsPostBack && !preventNotification)
        {
            DeferredMessage = "Save was Successful!";
        }
        else if (DeferredMessage != null)
        {
            SuppressSaveMessage = false;
            SaveSuccessMessage = DeferredMessage;
            RenderStartUpMessage(true);
            DeferredMessage = null;
        }
    }

    private void RestoreDateFromCache(string questionNumber, Dictionary<string, string> cache)
    {
        string dateString;
        if (cache.TryGetValue(questionNumber, out dateString))
        {
            dpDate1.SelectedDate = dpDate1.FocusedDate = DateTime.Parse(dateString);
        }
    }

    private static string DeferredMessage
    {
        get { return HttpContext.Current.Session[DeferredMessageKeyName] as string; }
        set { HttpContext.Current.Session[DeferredMessageKeyName] = value; }
    }

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        try
        {
            this.CheckActiveStatus(_surveyTypeId); // confirm instrument is active - if not, a redirect will occur.

            _tbiInstrumentManager.TbiServiceTo = base.GetTBIServiceTo();
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("VALowVisionSurvey.Page_Load(): _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }
            string instTitle = _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > " + instTitle;
            LabelInstrumentTitle.Text = instTitle;
            Page.Title = "TBI Instruments - " + instTitle;

            Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
            labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

            LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
            LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);
            if (!IsPostBack)
            {
                base.AddtoViewStateTBIServiceTo(_tbiInstrumentManager.TbiServiceTo);

                if (instrumentType.Status != null)
                {
                    _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
                }

                string _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
                if (!string.IsNullOrEmpty(_snum))
                {
                    string patICN = patientManager.GetPatientICNBySnum(_snum);

                    if (!string.IsNullOrEmpty(patICN))
                    {
                        ViewState["PatientICN"] = patICN;
                        RefreshSurvey(patICN);
                    }
                    else
                    {
                        throw new ArgumentException("Patient with this SSN does not exist.");
                    }

                    if (_survey != null && !_survey.isNew)
                    {
                        SetUpSurvey(_survey);
                    }
                    string referrer = "";
                    if (Request.UrlReferrer != null)
                    {
                        referrer = Request.UrlReferrer.PathAndQuery;
                    }
                    ViewState["referrer"] = referrer;
                }
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
            QuestionList = GetAllQuestionsForSurvey();
            SetQuestionLabels();

            dpDate1.MaxDate = DateTime.Now;
        }
        catch (Exception ex)
        {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "VALowVisionSurvey.aspx", "100027", "Page_Load()");
            Response.Redirect("Warning.aspx?err=2", false);
        }
    }

    private void SetQuestionLabels()
    {
        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        LabelDate1.Text = GetQuestionText("D");
        LabelForTYP.Text = GetQuestionText("TYP");
        LabelForQ1.Text = GetQuestionText("1");
        LabelForQ2.Text = GetQuestionText("2");
        LabelForQ3.Text = GetQuestionText("3");
        LabelForQ4.Text = GetQuestionText("4");
        LabelForQ5.Text = GetQuestionText("5");
        LabelForQ6.Text = GetQuestionText("6");
        LabelForQ7.Text = GetQuestionText("7");
        LabelForQ8.Text = GetQuestionText("8");
        LabelForQ9.Text = GetQuestionText("9");
        LabelForQ10.Text = GetQuestionText("10");
        LabelForQ11.Text = GetQuestionText("11");
        LabelForQ12.Text = GetQuestionText("12");
        LabelForQ13.Text = GetQuestionText("13");
        LabelForQ14.Text = GetQuestionText("14");
        LabelForQ15.Text = GetQuestionText("15");
        LabelForQ16.Text = GetQuestionText("16");
        LabelForQ17.Text = GetQuestionText("17");
        LabelForQ18.Text = GetQuestionText("18");
        LabelForQ19.Text = GetQuestionText("19");
        LabelForQ20.Text = GetQuestionText("20");
        LabelForQ21.Text = GetQuestionText("21");
        LabelForQ22.Text = GetQuestionText("22");
        LabelForQ23.Text = GetQuestionText("23");
        LabelForQ24.Text = GetQuestionText("24");
    }

    protected override void LoadViewState(object savedState) {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState() {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId)
    {
        foreach (STD_QUESTION ques in QuestionList)
        {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower())
            {
                return ques.QUESTION_TEXT;
            }
        }
        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber)
    {
        foreach (STD_QUESTION ques in QuestionList)
        {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower())
            {
                return ques.ID;
            }
        }
        return 0;
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e) {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        Response.Redirect(Request.Url.PathAndQuery, false);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e)
    {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        Page.Validate();
        SaveCompleteSurvey(ref status);

        if (Page.IsValid && status > 0)
        {
            preventNotification = true;
            Session["survey"] = _survey;
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        }
        else
        {
            Response.Redirect(Request.Url.PathAndQuery, false);
        }
    }

    /// <summary>
    /// Inactive timer firing.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void InActiveTimer_Tick(object sender, EventArgs e) {
        preventNotification = true;
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        Response.Redirect(Request.Url.PathAndQuery, false);
    }

    /// <summary>
    /// Cancel button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
    }

    /// <summary>
    /// Type of submission trigger
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void rblQTYP_SelectedIndexChanged(object sender, EventArgs e) {
        if (rblQTYP.SelectedItem.Text.ToLower() == BaselineText) {
            pnlBaselineScores.Visible = true;
            pnlFollowupScores.Visible = false;
        } else {
            pnlBaselineScores.Visible = true;
            pnlFollowupScores.Visible = true;
        }
    }

    /// <summary>
    /// On data bound to type selection
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void rblQTYP_DataBound(object sender, EventArgs e)
    {
        if (DoesPatientHaveABaseline())
        {
            pnlBaselineScores.Visible = true;
        }
    }

    /// <summary>
    /// Determine if patient has a baseline ... if not, force Baseline and disable follow-up.
    /// </summary>
    /// <returns></returns>
    protected bool DoesPatientHaveABaseline() {
        List<SURVEY> surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(_tbiInstrumentManager.TbiServiceTo.PatientICN, 17, false).ToList();
        bool hasBaseLine = false;
        bool hasCompleteBaseline = false;
        string averageBaselineScore = "";

        if (surveys != null) {
            foreach (SURVEY s in surveys) {
                foreach (SURVEY_RESULT sr in s.SURVEY_RESULTs.OrderBy(R => R.STD_QUESTION_CHOICE.STD_QUESTION.SORT_ORDER)) {
                    if (sr.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER == "TYP") {
                        if (sr.RESULT_TEXT.ToLower() == BaselineText)
                        {
                            hasBaseLine = true;
                            if (sr.SURVEY.SURVEY_STATUS.ToLower() == "completed") {
                                hasCompleteBaseline = true;
                            }
                        }
                    }

                    if (hasBaseLine) {
                        // get baseline score
                        if (sr.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER == "22") {
                            averageBaselineScore = sr.RESULT_TEXT;
                            txtQ22.Text = averageBaselineScore;
                        }
                    }
                }
            }
        }

        foreach (ListItem li in rblQTYP.Items) {
            if (li.Text.ToLower() == BaselineText)
            {
                // baseline
                if (hasBaseLine) {
                    if (hasCompleteBaseline) {
                        li.Enabled = false; 
                    } else {
                        li.Enabled = true;
                    }
                } else {
                    li.Enabled = true;
                }
            } else {
                // follow-up
                if (hasBaseLine) {
                    if (hasCompleteBaseline) {
                        li.Enabled = true;
                    } else {
                        li.Enabled = false;
                    }
                } else {
                    li.Enabled = false;
                }
            }
        }
        return hasBaseLine;
    }
    #endregion

    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status)
    {
        try
        {
            calculateScores();

            string patientICN = String.Empty;
            status = 0;
            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null)
            {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("VALowVisionSurvey.SaveCompleteSurvey(): WarningRedirect: String.IsNullOrEmpty(patientICN.Trim()");
                    //Lost state. Redirect to message that need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("VALowVisionSurvey.SaveCompleteSurvey(): WarningRedirect: ViewState[PatientICN] != null");
                //Lost state. Redirect to message that need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null)
            {
                if (_survey.isNew)
                {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("VALowVisionSurvey.SaveCompleteSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    }
                    else
                    {
                        _survey.SURVEY_STATUS = "inprocess";
                        _survey.SetAsInsertOnSubmit();
                        _survey.patient_ICN = patientICN;
                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);
                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);
                if (status != 0)
                {
                    //Fortify errors on this line (this.Master as RegistryInstrumentMaster).SetDirtyFlag(false);
                    ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);
                    RefreshSurvey(patientICN);
                    SuppressSaveMessage = false;
                }
            }
        }
        catch (Exception ex)
        {
            Exception _tbilogerror = HandleTBIException(ex, "VALowVisionSurvey.aspx", "100027", "SaveCompleteSurvey()");
        }
        finally
        {
            if (status == 0)
            {
                var cache = Helpers.SavePageToCache(this, FailedFormValuesName);
                ctrlDiagnosis.WriteToCache(cache);
                SaveDateToCache(dpDate1, "D", cache);
            }
            else
                Session[FailedFormValuesName] = null;//clear out the cached version on success
        }
    }

    private static void SaveDateToCache(Telerik.Web.UI.RadDatePicker date, string questionNumber, Dictionary<string, string> cache)
    {
        if (date.SelectedDate != null)
        {
            cache.Add(questionNumber, date.SelectedDate.Value.ToShortDateString());
        }
    }

    private void calculateQuestion(RadioButtonList radioButton, ref int questionCounter, ref int totalValue) 
    {

        if (radioButton.SelectedItem != null && radioButton.SelectedItem.Text != "0")
        {
            questionCounter++;
            totalValue += Convert.ToInt32(radioButton.SelectedItem.Text);
        }
    }

        /// <summary>
    /// Calculate scores
    /// </summary>
    private void calculateScores()
    {

        int numOfAnsweredQuestions = 0;
        int totalAnswerValues = 0;
        decimal averageScore = 0;

        if (rblQTYP.SelectedItem == null)
        {
            txtQ22.Text = "0";
            return;
        }
        if (rblQTYP.SelectedItem.Text.ToLower() == BaselineText)
        {
            #region Calculate Baseline Score

            calculateQuestion(rblQ2, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ3, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ4, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ5, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ6, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ7, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ8, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ9, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ10, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ11, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ12, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ13, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ14, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ15, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ16, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ17, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ18, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ19, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ20, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ21, ref numOfAnsweredQuestions, ref totalAnswerValues);

            if (numOfAnsweredQuestions != 0)
            {
                averageScore = Math.Round(Convert.ToDecimal(totalAnswerValues) / Convert.ToDecimal(numOfAnsweredQuestions), 2);
            }

            txtQ22.Text = averageScore.ToString();

            #endregion
        }
        else
        {
            #region Calculate Follow-Up Score

            calculateQuestion(rblQ2, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ3, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ4, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ5, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ6, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ7, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ8, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ9, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ10, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ11, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ12, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ13, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ14, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ15, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ16, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ17, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ18, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ19, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ20, ref numOfAnsweredQuestions, ref totalAnswerValues);
            calculateQuestion(rblQ21, ref numOfAnsweredQuestions, ref totalAnswerValues);

            if (numOfAnsweredQuestions != 0)
            {
                averageScore = Math.Round(Convert.ToDecimal(totalAnswerValues) / Convert.ToDecimal(numOfAnsweredQuestions), 2);
            }

            txtQ23.Text = averageScore.ToString();

            // determine overall change score ...
            if (!String.IsNullOrEmpty(txtQ22.Text))
            {
                decimal baselineScore = Convert.ToDecimal(txtQ22.Text);
                decimal overall = Math.Round(averageScore - baselineScore, 2);
                txtQ24.Text = overall.ToString();
            }

            #endregion
        }
    }

    private int SelectedDiagnosisID
    {
        get
        {
            string temp = this.ctrlDiagnosis.GetSelectedId();
            if (string.IsNullOrEmpty(temp))
                return 0;
            return Convert.ToInt32(temp);
        }
    }

    private void SaveCustomDiagnosis(SURVEY survey, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        string inputText = ctrlDiagnosis.GetSelectedResultText();
        int selectedID = SelectedDiagnosisID;
        if (null != currentResult)
        {
            if (currentResult.STD_QUESTION_CHOICE_ID != selectedID || currentResult.RESULT_TEXT != inputText)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.RESULT_TEXT = inputText;
                currentResult.STD_QUESTION_CHOICE_ID = selectedID;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else if (selectedID > 0)
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            result.RESULT_TEXT = inputText;
            result.STD_QUESTION_CHOICE_ID = selectedID;
            survey.SURVEY_RESULTs.Add(result);
        }
    }

    private void SaveDate(SURVEY survey, Telerik.Web.UI.RadDatePicker date, string questionNumber)
    {
        string inputText = "";
        if (date.SelectedDate != null)
        {
            inputText = ((DateTime)date.SelectedDate).ToShortDateString();
        }
        SaveText(survey, inputText, questionNumber);
    }

    private void SaveSingleSelect(SURVEY survey, RadioButtonList radioButtonList, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        int selectedValue = Convert.ToInt32(String.IsNullOrEmpty(radioButtonList.SelectedValue) ? "-1" : radioButtonList.SelectedValue);
        if (null != currentResult)
        {
            if (currentResult.STD_QUESTION_CHOICE_ID != selectedValue || currentResult.RESULT_TEXT != radioButtonList.SelectedItem.Text)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.STD_QUESTION_CHOICE_ID = selectedValue;
                currentResult.RESULT_TEXT = radioButtonList.SelectedItem.Text;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else if (selectedValue != -1)
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            survey.SURVEY_RESULTs.Add(result);
            result.STD_QUESTION_CHOICE_ID = selectedValue;
            result.RESULT_TEXT = radioButtonList.SelectedItem.Text;
        }
    }

    private void SaveText(SURVEY survey, string text, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        if (null != currentResult)
        {
            if (currentResult.RESULT_TEXT != text)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.RESULT_TEXT = text;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else if (text != string.Empty)
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            result.RESULT_TEXT = text;
            result.STD_QUESTION_CHOICE_ID = availableChoices.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            survey.SURVEY_RESULTs.Add(result);
        }
    }

    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        string questionNumber = "";
        try
        {
            //first, mark all current answers for deletion
            survey.SetAsChangeTrackingRoot();
            foreach (SURVEY_RESULT tempResult in survey.SURVEY_RESULTs)
                tempResult.SetAsDeleteOnSubmit();

            SaveCustomDiagnosis(survey, questionNumber = "DX");
            SaveDate(survey, dpDate1, questionNumber = "D");
            SaveSingleSelect(survey, rblQTYP, questionNumber = "TYP");
            SaveSingleSelect(survey, rblQ1, questionNumber = "1");
            SaveSingleSelect(survey, rblQ2, questionNumber = "2");
            SaveSingleSelect(survey, rblQ3, questionNumber = "3");
            SaveSingleSelect(survey, rblQ4, questionNumber = "4");
            SaveSingleSelect(survey, rblQ5, questionNumber = "5");
            SaveSingleSelect(survey, rblQ6, questionNumber = "6");
            SaveSingleSelect(survey, rblQ7, questionNumber = "7");
            SaveSingleSelect(survey, rblQ8, questionNumber = "8");
            SaveSingleSelect(survey, rblQ9, questionNumber = "9");
            SaveSingleSelect(survey, rblQ10, questionNumber = "10");
            SaveSingleSelect(survey, rblQ11, questionNumber = "11");
            SaveSingleSelect(survey, rblQ12, questionNumber = "12");
            SaveSingleSelect(survey, rblQ13, questionNumber = "13");
            SaveSingleSelect(survey, rblQ14, questionNumber = "14");
            SaveSingleSelect(survey, rblQ15, questionNumber = "15");
            SaveSingleSelect(survey, rblQ16, questionNumber = "16");
            SaveSingleSelect(survey, rblQ17, questionNumber = "17");
            SaveSingleSelect(survey, rblQ18, questionNumber = "18");
            SaveSingleSelect(survey, rblQ19, questionNumber = "19");
            SaveSingleSelect(survey, rblQ20, questionNumber = "20");
            SaveSingleSelect(survey, rblQ21, questionNumber = "21");
            SaveText(survey, txtQ22.Text, questionNumber = "22");
            SaveText(survey, txtQ23.Text, questionNumber = "23");
            SaveText(survey, txtQ24.Text, questionNumber = "24");
        }
        catch (Exception surveyEx)
        {
            Exception _tbilogerror = HandleTBIException(surveyEx, "VaLowVisionSurvey.aspx", "100027", "SaveSurvey() - Question " + questionNumber);
            throw;
        }
    }// END SaveSurvey

    private void SetCustomDiagnosisElement(SURVEY_RESULT result, ASP.controls_tbidiagnosisselector_ascx diagControl)
    {
        if (result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
        {
            diagControl.SelectedChoiceId = result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
            if (result.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other"))
            {
                diagControl.SetOtherDescriptionText(result.RESULT_TEXT);
            }
        }
        else
        {
            diagControl.SelectedChoiceId = null;
        }
    }

    private void SetDateElement(SURVEY_RESULT result, Telerik.Web.UI.RadDatePicker date)
    {
        if (!String.IsNullOrEmpty(result.RESULT_TEXT))
        {
            DateTime temp;
            if (DateTime.TryParse(result.RESULT_TEXT, out temp))
            {
                date.SelectedDate = temp;
            }
        }
    }

    private void SetRadioElement(SURVEY_RESULT result, RadioButtonList radioButton)
    {
        if (result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
        {
            radioButton.SelectedValue = result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
        }
        else
        {
            radioButton.SelectedIndex = -1;
        }
    }

    private void SetTextBoxElement(SURVEY_RESULT result, TextBox textBox)
    {
        if (!String.IsNullOrEmpty(result.RESULT_TEXT))
        {
            textBox.Text = result.RESULT_TEXT;
        }
    }

    private void SetTYPElement(SURVEY_RESULT result, RadioButtonList radioButton)
    {
        if (result.RESULT_TEXT.ToLower() == BaselineText)
        {
            pnlBaselineScores.Visible = true;
            pnlFollowupScores.Visible = false;
        }
        else
        {
            pnlBaselineScores.Visible = true;
            pnlFollowupScores.Visible = true;
        }
        SetRadioElement(result, radioButton);
    }

    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey)
    {
        try
        {
            foreach (var sResult in survey.SURVEY_RESULTs)
            {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER)
                {
                    case "DX": SetCustomDiagnosisElement(sResult, ctrlDiagnosis); break;
                    case "D": SetDateElement(sResult, dpDate1); break;
                    case "TYP": SetTYPElement(sResult, rblQTYP); break;
                    case "1": SetRadioElement(sResult, rblQ1); break;
                    case "2": SetRadioElement(sResult, rblQ2); break;
                    case "3": SetRadioElement(sResult, rblQ3); break;
                    case "4": SetRadioElement(sResult, rblQ4); break;
                    case "5": SetRadioElement(sResult, rblQ5); break;
                    case "6": SetRadioElement(sResult, rblQ6); break;
                    case "7": SetRadioElement(sResult, rblQ7); break;
                    case "8": SetRadioElement(sResult, rblQ8); break;
                    case "9": SetRadioElement(sResult, rblQ9); break;
                    case "10": SetRadioElement(sResult, rblQ10); break;
                    case "11": SetRadioElement(sResult, rblQ11); break;
                    case "12": SetRadioElement(sResult, rblQ12); break;
                    case "13": SetRadioElement(sResult, rblQ13); break;
                    case "14": SetRadioElement(sResult, rblQ14); break;
                    case "15": SetRadioElement(sResult, rblQ15); break;
                    case "16": SetRadioElement(sResult, rblQ16); break;
                    case "17": SetRadioElement(sResult, rblQ17); break;
                    case "18": SetRadioElement(sResult, rblQ18); break;
                    case "19": SetRadioElement(sResult, rblQ19); break;
                    case "20": SetRadioElement(sResult, rblQ20); break;
                    case "21": SetRadioElement(sResult, rblQ21); break;
                    case "22": SetTextBoxElement(sResult, txtQ22); break;
                    case "23": SetTextBoxElement(sResult, txtQ23); break;
                    case "24": SetTextBoxElement(sResult, txtQ24); break;
                    default: break;
                }
            }
        }
        catch (Exception ex)
        {
            Exception _tbilogerror = HandleTBIException(ex, "VALowVisionSurvey.aspx", "100027", "SetUpSurvey()");
        }
    }// END SetUpSurvey

    #endregion
}